#version 130

const float PI = 3.14159265359;
const float TWOPI = 2.0 * PI;
const float PI_OVER_2 = PI / 2.0;
const float EPSILON = 0.000001;
const float SQRT_2 = 1.4142;

//List of textures, which are defined by a single array in Java TEXT_INDEX_COLOR, TEXT_INDEX_DEPTH,:
uniform sampler2D u_texPano; //The color image, RGB only?
uniform sampler2D u_texDepth; //The depth buffer with RGBA info
uniform sampler2D u_texHigh; //The current mip-map block of the full-sized color image corresponding to this pixel's "mesh".
uniform sampler2D u_texPatchMask; //The demolition patch preview, 16-bit uShort per pixels
uniform sampler2D u_texPatchColor; //The demolition patch ARGB color + intensity texture
uniform sampler2D u_texPatchDepth; //The demolition patch ARGB depth texture

uniform float u_fAmplitude; //0-1 for blinking demolition patch
uniform float u_fHighTexMix; //Fade to high res (u_texHigh), 0 = fully u_texPano, whereas 1 = fully u_texHigh.
uniform float u_fHasMaskZ; //0/1 boolean float indicating whether u_texPatchMask is defined.
uniform float u_fHasPatchColor; //0/1 boolean float indicating whether u_texPatchColor is defined.
uniform float u_fHasPatchDepth; //0/1 boolean float indicating whether u_texPatchDepth is defined.
uniform float u_fShowPatchRGB; //0/1 boolean float indicating whether u_texPatchColor should show ONLY Alpha (intensity) or only RGB (color) channels.

uniform float u_fDoDepth; //0/1 boolean float indicating whether "Rendering Mode" is OFF/ON

uniform float u_fFOV; //The current Field of View's horizontal angle in radians of the current view.
uniform ivec2 u_iv2FrameSize; //[horizontal,vertical] span pixel counts.
uniform float u_fNearClip; //The closest distance from camera that will be rendered
uniform float u_fFarClip; //The farthest distance from camera that will be rendered

//Image color, u_fContrast, luminosity, u_fBrightness, etc. adjustments:
uniform float u_fIntensityMin; //Rescale the intensity so that this is the lowest possible value.
uniform float u_fIntensityMax; //Rescale the intensity so that this is the highest possible value.
uniform float u_fBrightness; // Sane values are +/-1.2
uniform float u_fContrast; // 0.0 : none, 25.0 : burned
uniform float u_fSaturation; //0.0 : b&w, 25: heatmap.

out vec4 o_v4FragColor; //The final output pixel ARGB that will be displayed on screen

/**@return The angle between the forward direction and the direction from the eye to the input pixel indices.*/
float getAngleFromCenter(vec2 pixelPosition){
  vec2  fromCenter = vec2(pixelPosition.x - u_iv2FrameSize.x / 2, pixelPosition.y - u_iv2FrameSize.y / 2);
  float   e = length(fromCenter);
  return atan( ( e * 2 * tan(u_fFOV / 2) ) / u_iv2FrameSize.x );
}

/**@return The near-plane normal distance to the point defined by length d and direction pixelPosition.*/
float transformDepth(float d, vec2 pixelPosition){
  float THETA = getAngleFromCenter(pixelPosition); //Gives d* from d via an arctangent calculation.
  return d * cos (THETA); 
}

/**
 * @param value A distance between "near" and "far".
 * @return A value between 0-1, which is logarithmicly scaled.
 */
float zDepthFromDistanceMilli(float value, float near, float far){
  float nearInv = 1 / near;
  float farInv = 1 / far;
  return ( (1 / value) - nearInv ) / (farInv - nearInv);
}

/**
 * @param color An RGB that contains the angle offsets and depth distance compressed.
 * @return Only the depth distance contained in the pixel.
 */
int colorToMillimeters(vec3 color){
	int red   = int(255*color.r);
	int green = int(255*color.g);
	int blue  = int(255*color.b);	
	return int(red/100)*256*256 + green*256 + blue;
}

/**
 * @return Interpolation between 4 neighboring pixels for the compressed depth buffer,
 *(which cannot simply be interpolated like colors can).
 */
float extractDepth (sampler2D texture, vec2 texCoord){
	ivec2 size = textureSize(texture, 0);

	vec2 depthCoord = texCoord * vec2(size);
	ivec2 idepthCoord = ivec2( floor(depthCoord) ); // integer texel position

    ivec4 dcsquare = ivec4(idepthCoord.xy, mod((idepthCoord.x +1), size.x), 0);
    dcsquare.w = idepthCoord.y + int(step(float(size.y), idepthCoord.y +1.0));//()

    depthCoord -= idepthCoord;//just the decimal part

	vec4 depthPick[4] = vec4[](
		texelFetch(texture, dcsquare.xy, 0).rgba, // 
		texelFetch(texture, dcsquare.zy, 0).rgba, // 0 1
		texelFetch(texture, dcsquare.xw, 0).rgba, // 2 3
		texelFetch(texture, dcsquare.zw, 0).rgba
	); //
	float depths[4]; //The 4 distances
	float fDepthMilli; //The output return value
	for(int i = 0; i < 4; ++i){ //Calculate the depth distance at all 4 neighboring pixels
		int d = colorToMillimeters(depthPick[i].rgb);
		depths[i] = mix( 0.0, float(d), (d>0) );
	}

	//TODO:optim
	float d1;//The horizontal average distance between top-left and top-right pixels 
	if(depths[0] == 0.0) //No point exists
		d1 = depths[1];
	else if(depths[1] == 0.0) //No point exists
		d1 = depths[0];
	else
		d1 = mix(depths[0], depths[1], depthCoord.x);//Average horiz.
	float d2;//The horizontal average distance between bottom-left and bottom-right pixels 
	if(depths[2] == 0.0) //No point exists
		d2 = depths[3];
	else if(depths[3] == 0.0) //No point exists
		d2 = depths[2];
	else
		d2 = mix(depths[2], depths[3], depthCoord.x);

	if(d2 == 0.0) //Neither lower points exist
		fDepthMilli = d1;
	else if(d1 == 0.0) //Neither upper points exist
		fDepthMilli = d2;
	else //Average of the top-ave and bottom-ave
		fDepthMilli = mix(d1, d2, depthCoord.y);

	return fDepthMilli;
	//return mix(d1, d2, depthCoord.y);
}

/**
 * @param The desired u_fBrightness offset between +/-1.0
 * @return Translation matrix to change the u_fBrightness of an RGB1 vector4.
 */
mat4 brightnessMatrix( float u_fBrightness ) {
    return mat4( 1, 0, 0, 0,
                 0, 1, 0, 0,
                 0, 0, 1, 0,
                 u_fBrightness, u_fBrightness, u_fBrightness, 1);
}
/**
 * @param The desired u_fContrast between 0-1.
 * @return Generates a transformation matrix that can be applied to an RGB1 vec4 to change its u_fContrast.
 */
mat4 contrastMatrix( float u_fContrast ) {
	float t = ( 1.0 - u_fContrast ) / 2.0;
	//4x4 Scale matrix:
    return mat4( u_fContrast, 0, 0, 0,
                 0, u_fContrast, 0, 0,
                 0, 0, u_fContrast, 0,
                 t, t, t, 1 );
}
/**
 * @param The desired u_fSaturation between 0-1.
 * @return Generates a transformation matrix that can be applied to an RGB1 vec4 to change its u_fSaturation.
 */
mat4 saturationMatrix( float u_fSaturation )
{
    vec3 luminance = vec3( 0.3086, 0.6094, 0.0820 );
    
    float oneMinusSat = 1.0 - u_fSaturation;
    
    vec3 red = vec3( luminance.r * oneMinusSat );
    red+= vec3( u_fSaturation, 0, 0 );
    
    vec3 green = vec3( luminance.g * oneMinusSat );
    green += vec3( 0, u_fSaturation, 0 );
    
    vec3 blue = vec3( luminance.b * oneMinusSat );
    blue += vec3( 0, 0, u_fSaturation );
    //4x4 Rotation&scale matrix:
    return mat4( red,     0,
                 green,   0,
                 blue,    0,
                 0, 0, 0, 1 );
}
/**
 * @return True if the point is contained inside the specified 3D Bounding-Box (min, max).
 */
bool inside(vec3 point, vec3 minBB, vec3 maxBB)
{
	bvec3 overMin  = lessThan(minBB, point);
	bvec3 underMax = lessThan(point, maxBB);
	return all(overMin) && all(underMax);
	//return any(not(overMin)) || any(not(underMax));
}
/**
 * Entry point for the shader code.
 */
void main()
{
	//vec4 v4ColorTex; //The RGB color pixel at the current coordinate.
	//vec4 v4ColorDepth;
	vec4 v4ColorPick; //?

	ivec2 iv2DepthSize = textureSize(u_texDepth, 0); //The width and height of the depth buffer

	float fDistMilli = 0;
	float fDepthMilli = 0;
	bool bCalculateDepth = u_fDoDepth > 0.5; //True if the depth should be computed for z-order
	float fSphereDepth = gl_FragCoord.z;

	//TODO : understand why we need to offset picking coordinates!
	vec2 v2DepthShift = vec2(-1.0, 0.5);

	vec2 tcColor = gl_TexCoord[0].xy; //The current pixel's coordinate in the color image?
	vec2 tcDepth = gl_TexCoord[1].xy; //The current pixel's coordinate in the depth buffer image?
	vec2 tcHigh = clamp(gl_TexCoord[2].xy, vec2(0, 0), vec2(1, 1)); //?

	bool bPatchThisPixel = (u_fHasMaskZ == 1.0) || u_fHasPatchDepth == 1.0;
	if (bPatchThisPixel) { //Will this pixel potentially be overwritten from a patch?
		if (u_fHasMaskZ == 1.0){ //If we are given a mask that tells us which pixels are part of the demolition, we can use it directly here
			float fMask = texture2D(u_texPatchMask, tcColor).r;
			bPatchThisPixel = fMask != 0.f;
		}
		else { //Otherwise, we can try to determine which pixels are valid by their depth buffer value.
			ivec2 tcLocal = ivec2(textureSize(u_texPatchDepth, 0)*tcColor);
			vec3 v3Mask = texelFetch(u_texPatchDepth, tcLocal, 0).rgb; //If alpha is NOPOINT=254, assume no data (NOPOINT quality)
			bPatchThisPixel = (v3Mask != vec3(0.0)); //The universal "NO DATA" sign is when the RGB depth is 0, but doesn't use the quality (in alpha channel) at all
		}
	}
	v4ColorPick = texture2D(u_texPano, tcColor).rgba; // pick color??
	if (bPatchThisPixel) {
		if (u_fHasPatchColor == 1.0) { //Get the RGB color from the patch
			ivec2 tcLocal = ivec2(textureSize(u_texPatchColor, 0)*tcColor);
			//vec3 patchGraySample = texelFetch(u_texPatchColor, tcLocal, 0).aaa;
			vec3 v3PatchColorSample;
			if (u_fShowPatchRGB == 1.f)
				v3PatchColorSample = texelFetch(u_texPatchColor, tcLocal, 0).rgb; //Get the RGB color from: Sampler, texture coordinate, level-of-detail
			else
				v3PatchColorSample = texelFetch(u_texPatchColor, tcLocal, 0).aaa; //Get the grayscale intensity from alpha: Sampler, texture coordinate, level-of-detail
			if (v3PatchColorSample != vec3(0.0))
				v4ColorPick = mix(v4ColorPick, vec4(v3PatchColorSample, 1.0), u_fAmplitude);
		}
		else //Show a temporary color while demolition occurs
			v4ColorPick = vec4(1.0,0.0,0.0,1.0); //The universal "NO DATA" sign is when the RGB depth is 0, but doesn't use the quality (in alpha channel) at all
	}
	else
		v4ColorPick = texture2D(u_texPano, tcColor).rgba; // pick color
	if (bCalculateDepth) {
		tcDepth += v2DepthShift/vec2(iv2DepthSize);
		fDistMilli = extractDepth(u_texDepth, tcDepth);
		if (bPatchThisPixel) {
			float distMilliPatch; 
			if (u_fHasPatchDepth == 1.0) //Show the depth from the patch
				distMilliPatch = extractDepth(u_texPatchDepth, tcDepth);
			else //Temporarily show a far distance
				distMilliPatch = u_fFarClip*1000.0;
			float ratio = smoothstep(0.2,0.8,u_fAmplitude);
			fDistMilli = mix(fDistMilli, distMilliPatch, ratio);
		}
		fDepthMilli = transformDepth(fDistMilli, gl_FragCoord.xy);
	}
	
	fDepthMilli *= 1.0055;// correct "fingerprint" effect ?

	vec4 v4ColorHighPick;
	if (textureSize(u_texHigh, 0).x > 0)
		v4ColorHighPick = texture2D(u_texHigh, tcHigh).rgba;
	else
		v4ColorHighPick = vec4(0.0);
	//highColor = v4ColorHighPick;
	vec4 v4ColorFinal;
	if (bPatchThisPixel)
		v4ColorFinal = v4ColorPick; //Use the patch color or temp red.
	else
		v4ColorFinal = mix(v4ColorPick, v4ColorHighPick, u_fHighTexMix); //Show the regular un-patched color
	o_v4FragColor = 1.0 //Transform the color by the brightness, contrast, saturation, and intensity clamp, all specified by user.
		  * brightnessMatrix(u_fBrightness)
          * contrastMatrix(u_fContrast)
          * saturationMatrix(u_fSaturation)
          * vec4( clamp( (v4ColorFinal.rgb - u_fIntensityMin) / (u_fIntensityMax - u_fIntensityMin), 0.0, 1.0 ) , 1.0);
	if (bCalculateDepth) {
		float fDepth = zDepthFromDistanceMilli(float(fDepthMilli/1000), 1.0 * u_fNearClip, 1.0 * u_fFarClip);

		
		float fDist = mix ( float(fDistMilli/1000), fSphereDepth, (fDistMilli <= 10));
		vec2 sphericalCoord = vec2( 1.0 - tcDepth.x, -tcDepth.y) * vec2(TWOPI, PI) + vec2(0, PI_OVER_2);
		float   x = cos(sphericalCoord.x) * cos(sphericalCoord.y),
				y = sin(sphericalCoord.x) * cos(sphericalCoord.y),
				z = sin(sphericalCoord.y);

		vec4 myPointPos = vec4(vec3(x, y, z) * fDist, 1.0);
		vec4 myPointVec = vec4(vec3(x, y, z) * fDist, 0.0);

		bool bInsideAny = false;//myPointPos.y > 1.0 && myPointPos.y < 1.1;

		/*
		if(numDemoBBS > 0){
			for(int i = 0; i < numDemoBBS; ++i){
				mat4 mtx = mtxDemoBBS[i] ;
				mat3 rot = mat3(mtx);
				vec4 trans = mtx[3].xyzw;
				vec3 pt = rot * (myPointVec.xyz) + trans.xyz ;
				vec4 min = vec4( minDemoBBS[i], 1.0) ;
				vec4 max = vec4( maxDemoBBS[i], 1.0) ;
				bool insideThis = inside(pt.xyz, min.xyz, max.xyz) && false;
				bInsideAny = bInsideAny || insideThis || (pt.y > 1.89 && pt.y < 1.99);
			}
		}//*/
		bInsideAny = false;
		gl_FragDepth = mix(fSphereDepth, fDepth, (fDepthMilli >= 10 && !bInsideAny));
		o_v4FragColor = mix(o_v4FragColor, vec4(1.0,0.0,0.0,0.0), float(bInsideAny));
	}
	else {
		gl_FragDepth = fSphereDepth;
	}
}
