

#ifndef __OPENCL_VERSION__ //This should fail during an actual openCL compile, used only to trick Eclipse into syntax highlighting this file as "C" code.
#define __kernel
#define __global
#define kernel
#define global
#define constant
#define local
#define float2 float
#define float3 float
#define float4 float
#define uchar4 char
#endif

/*----------------------------------------------------------*//**
* Determine if a 3D coordinate is contained within a spherical volume.
* @param point The 3D point to check whether inside this single volume.
* @param sphere The x,y,z,r of a sphere.
* @return 1/0 if point is/isn't contained in the single volume,
*//*-----------------------------------------------------------*/
bool containedInSphere1(const float3 point, const float4 sphere) {
	float3 disp = point.xyz - sphere.xyz;//Get point position relative to the sphere's center
	return (sphere.w*sphere.w >= dot(disp.xyz, disp.xyz));//Point lies within spherical radius?
	//return dot(globalCoords.xyz, globalCoords.xyz)*20; //Distance from station
	//return dot(disp.xyz, disp.xyz); //Distance from sphere
}

/*----------------------------------------------------------*//**
* Determine farthest distance that a unit vector "direction" emitted from "eye" intersects with the "sphere" volume.
* @param eye The 3D point which emits the unit vector.
* @param direction A UNIT vector defining the direction of the intersecting line to check.
* @param sphere The x,y,z,r of a sphere.
* @return The farthest distance to the intersection of the line and the volume, INFINITY if none found.
*//*-----------------------------------------------------------*/
float intersectsSphere(const float3 eye, const float3 direction, const float4 sphere) {
	const float3 d = eye - sphere.xyz;//Eye relative to the sphere center relative
	const float3 u = direction; //Ray direction from the camera
	const float ud = dot(u.xyz, d.xyz);
	const float dd = dot(d.xyz, d.xyz);
	const float disc = ud*ud - dd + sphere.w*sphere.w;//Discriminant

	if (disc < 0.f) //Imaginary discriminant
		return INFINITY; //No solutions, not *intersecting volume*
	//else if (disc >= 0.f) { //1-2 solutions, but we must return the one farthest positive from camera eye.
	float sqrtdisc = sqrt(disc);
	const float lam1 = -ud-sqrtdisc; //smaller
	const float lam2 = -ud+sqrtdisc; //larger

	if (lam1 > 0)
		return lam2;
	else if (lam2 < 0)
		return INFINITY;
	return FLT_MIN;//Solution is exactly zero, so return a slightly positive result
	//Negative distances are in the opposite direction of the camera
	//(actually we are redundantly calculating this twice over a 360 scanorama)
	//Try using intersectsSphere2 to capture both points simulataneously.
}

/*----------------------------------------------------------*//**
* Check for point containment or line intersection with a sphere.
* @param p Buffer stream index, is updated within.
* @param volumeParamBuffer A raw byte stream containing information about all volumes serialized (concatenated) together.
* @param coords [contains] The 3D point (local to station) to check inside this single volume.
* 					  [intersection] The unit direction vector emanating from point coords.
* @param direction null (0) if checking point, otherwise the direction vector to check for ray intersection (starting at origin coords)
* @return [contains] [1/0,0] if point is/isn't contained in the single volume, Only the float2.x is used, float2.y is UNUSED.
*		  [intersection] 2 distances [0,180] degrees from "direction", always positive distance (d>0) to the rear face of the volume,
*		  or INFINITY if NOT intersecting the ray.
*//*-----------------------------------------------------------*/
float touchesSphere(uint *p, constant float *volumeParamBuffer, const float3 coords, const float3 *direction) {
	//unpack the serialized stream:
	float4 sphere = readFloat4(p, volumeParamBuffer);
	//Read 4 floats at 4 bytes each from the "volume description" stream
	if (direction == 0) { //Need to check if a *point is contained within the volume*:
		return containedInSphere1(coords, sphere) ? 1.f : 0.f;//1/0 for contained or not
	}
	//Else, need to check if a *line intersects the volume*:
	return intersectsSphere(coords, *direction, sphere);
}

/*----------------------------------------------------------*//**
* Check for point containment (if direction==NULL) or line intersection (if direction!=NULL) with a list of volumes.
* Returns the distance to the ray intersection of the rear-face of the closest volume.
* If the point lies within any of the volumes, then we must get the distance of the rear-face closest
* @param volumeTypes Stream of bytes describing total number of volumes, followed by an ID for each volume.
* @param volumeParamBuffer A float stream containing floats/ints (all 4 byte aligned, very important!) information about all volumes serialized (concatenated) into a single "stream" buffer.
* @param globalCoords The 3D point to check if it resides inside the volumes list., or the eye of the ray tracing
* @param globalDirection NULL for contains operation, otherwise the direction of the ray emitted from the eye (at globalCoords).
* @return depends whether it was a contains or intersect operation (see input arguments):
* 	 0.0: [Contains] The point exists in the scanorama, but was NOT contained in any of the volumes.
* 	 1.0: [Contains] The point is contained in at least one of the volumes.
*	>0.0: [Intersect] Distance to the rear surface of the closest(?) volume intersection with ray (from eye to existing scanaorama point).
* 	+Inf: [Intersect] No intersection with any volumes.
* 	 NAN: ERROR: unknown volume shape.
*//*-----------------------------------------------------------*/
float touchesVolume(constant float *volumeParamBuffer, const float3 coords, const float3 *direction) {
	uint p = 0; //Initially point to the START of volumeParamBuffer to step through the stream.
	int numVols = readInt(&p, volumeParamBuffer); //Read first int, containing number of shapes (1-127).
	if (numVols < 1 || numVols > 30) //TODO: remove max cap
		return NAN;//Error: May have read the buffer incorrectly
	float fFinal, fOut;
	if (direction != NULL) //"Intersect" operation only
		fFinal = fOut = INFINITY; //Nearest volume's distance to its rear-most face intersection.
	else
		fFinal = fOut = 0.f; //"Contains" operation fails by default

	for (int v = 0; v < numVols; v++) { //Cycle through all volumes (1-based index), reading the volumeParamBuffer stream
		int volType = readInt(&p, volumeParamBuffer); //read 32-bit integer, containing type (class_id) of current shape
		switch (volType) {
			case 3408: //Sphere:
				fOut = touchesSphere(&p, volumeParamBuffer, coords, direction);
				break;
			case 2: //Other TODO:
				fOut = 41; //touchesX(p, volumeParamBuffer, coords, direction);
				break;
			default: { //Error: Invalid volume type, stop immediately.
				continue; //fOut = NAN;
			}
		}
		if (direction == NULL) { //"Contains" operation
			if (fOut == 1.f) //It was contained in the current volume, break immediately, don't search further volumes
				return fOut;
		}
		else { //"Intersects" operation
			if (fOut < fFinal) //If it's closer, save the newest/closest one
				fFinal = fOut;
		}
	}
	return fFinal; //"Contains" found no volumes, -OR- "Intersection" operation returns closest intersection distance after cycling through ALL volumes
}

/*----------------------------------------------------------*//**
* The input 2D image containing buffer depth information will be "intersected" with the input volume list, returning a mask image describing whether
* the volumes were contained, intersected (at what distance), or neither.
* @param destFlagBuffer The 2D output mask describing the results of the contain() operation for each "pixel" in the image.
* @param srcDepthBuffer The 2D image to check for points contained in volume list specified by volumeParamBuffer.
* @param volumeParamBuffer A byte stream containing information about all volumes serialized (concatenated) into a single "stream" buffer.
* @param canvasRecords (UNUSED) output point records.
* @param srcMtx a matrix transformation related to the station origin/direction?
* @param depthW The number of pixels horizontally across the image.
* @param depthH The number of pixels vertically across the image.
* @return (output written to destFlagBuffer):
* 	   0: The point exists in the scanorama, but was NOT contained in any of the volumes.
* 	+Inf: ERROR: a point was contained in a volume, BUT has no intersect with any volumes. May be round-off error between contains/intersects methods?
*	-Inf: No pixel info was in the scanorama, and there was no intersection with any of the volumes
*	 NAN: The volume shape was not a known type!
*	  >0: Distance to the rear surface of the closest(?) volume intersection with ray (from eye to existing scanaorama point).
*	  <0: Distance to the rear surface of the closest(?) volume intersection with ray (from eye to estimated non-existing scanaorama angular coordinate).
*//*-----------------------------------------------------------*/
kernel void identifyImpactedPixels(global float *destFlagBuffer, global const uchar4 *srcDepthBuffer, constant float *volumeParamBuffer, global struct stPixelRecord *canvasRecords,
		constant float *srcMtx, const int depthW, const int depthH)
{ //can't have constant/global on scalar inputs
	int global_id = get_global_id(0);
	int row = global_id / depthW; //convert global 1D index into 2D
	int col = global_id % depthW; //convert global 1D index into 2D
	if (row >= depthH) //skip if out of bounds, may happen when rounding-up to achieve an image size evenly-divisible by group-size.
		return;

	uchar4 depthInfo = srcDepthBuffer[global_id];
	float3 localCoords = getDepthBufferLocalCoordinates(depthInfo, col, row, depthW, depthH);//xyz coordinates of the point relative to the station
	short iExists = (localCoords.x == 0.f && localCoords.y == 0.f && localCoords.z == 0.f) ? -1 : 1;//+/-1 if the point exists/doesn't in the scanorama's pixel
	if (iExists > 0) { //Determine if this point coordinate is contained in ANY of the volumes
		float3 globalCoords = transformPoint_arrayVersion(srcMtx, localCoords);//Convert local to global coordinates
		destFlagBuffer[global_id] = touchesVolume(volumeParamBuffer, globalCoords, NULL);//determine if contained in list of 3D volumes
		if (destFlagBuffer[global_id] == 0.f) //If true (0), the point was NOT contained in any of the volumes,
			return; // So skip it, (no overwriting on the scanorama's pixel necessary).
	}
	//else, no point exists at the pixel.
	//Either the point was inside a volume OR there was no data: iPointExists=+/-1, so we need to know the distance to the rear face of the volume.
	float3 localDirection  = getLocalDirection_Panoramic((float)col+0.5f, (float)row+0.5f, depthW, depthH);//Unit vector direction from eye
	float3 globalDirection = transformVector_arrayVersion(srcMtx, localDirection);//Convert local to global
	float3 zero = (float3)(0.f); //Zero vector: origin in local coordinates
	float3 origin = transformPoint_arrayVersion(srcMtx, zero); //Origin of station in global coordinates: station "eye".
	destFlagBuffer[global_id] = iExists * touchesVolume(volumeParamBuffer, origin, &globalDirection);//Check if the line intersects any of the volumes, get the (negative) distance
}
/**
Now we get multiple lists corresponding to the list of points from the octree, buffimage index. Then we check the intersection of the mask with each point to find corresponding ones
*/
/*kernel void flattenPointsIntoPatch(global const float4 *points, global const float *destFlagBuffer, const int depthW, const int depthH) {
	;
}*/
