#ifndef __OPENCL_VERSION__ //This should fail during an actual openCL compile, used only to trick Eclipse into syntax highlighting this file as "C" code.
#define __kernel
#define __global
#define kernel
#define global
#define constant
#define local
#define float2 float
#define int3 int
#define float3 float
#define float4 float
#define uchar4 char
#endif

#define circleParamIndexX 0
#define circleParamIndexY 1
#define circleParamIndexR 2

#define dLambdaInitial 0.001
#define thresholdDeltaF 0.001
#define lambdaScaleFactor 10
/** Max number if iterations for Hessienne. */
#define ITER_MAX 20

float deriveeXCercleFast(float x, float y, float cx, float cy, float r, float D)
{		
	return (cx-x)/D;
}
float deriveeYCercleFast(float x, float y, float cx, float cy, float r, float D)
{		
	return (cy-y)/D;
}
float deriveeRCercle(float x, float y, float cx, float cy, float r)
{		
	return -1;
}
float distanceSigneeFastCircle(float x, float y, const float *parametres, float D) 
{
	return D - parametres[circleParamIndexR];
}
float distanceToCenterCircle(float x, float y, const float *parametres)
{
	float cx = parametres[circleParamIndexX];
	float cy = parametres[circleParamIndexY];
	return sqrt((x-cx)*(x-cx)+(y-cy)*(y-cy));
}
float getSquareDistanceCircle(float x, float y, const float *parametres)
{
	float D = distanceToCenterCircle(x, y, parametres);
	float dist = distanceSigneeFastCircle(x, y, parametres, D);
	return dist * dist;
}
float deriveePartielleFastCircle(int i, float x, float y, const float *parametres, float D)
{
	float cx = parametres[circleParamIndexX];
	float cy = parametres[circleParamIndexY];
	float r = parametres[circleParamIndexR];
	float result;
	if (i==circleParamIndexX)
	{
		result = deriveeXCercleFast(x, y, cx, cy, r, D);
		return result;
	}
	if (i==circleParamIndexY)
	{
		result = deriveeXCercleFast(x, y, cx, cy, r, D);
		return result;
	}
	if (i==circleParamIndexR)
	{
		result = deriveeRCercle(x, y, cx, cy, r);
		return result;
	}
	return 0;
}
float calculateSumMeanSquaresCircle(uchar* topologyGrid, const uchar formId, const int nbCellsX, const int nbCellsY, const float *parametres)
{
	float somme = 0;
	for (int ix=0; ix<nbCellsX; ix++)
	{
		for (int iy=0; iy<nbCellsY; iy++)
		{
			int ind = iy*nbCellsX+ix;
			if (topologyGrid[ind] == formId)
				somme += getSquareDistanceCircle(ix, iy, parametres);
		}
	}
	return somme;
}


void hessienne(float x, float y, const float *parametres, float *H)
{
	// Matrice des dérivées seconde
	// Approximation valable au voisinage de la solution (on ne prend en compte que les dérivées premières)
	float deriveesPartielles[3] = {0};
	float D = distanceToCenterCircle(x, y, parametres);
	for (int i=0; i<3; i++)
		deriveesPartielles[i] = deriveePartielleFastCircle(i, x, y, parametres, D);
		
	for (int i=0; i<3; i++)
	{
		for (int j=i; j<3; j++)
		{
			H[i*3+j] = 2 * deriveesPartielles[i] * deriveesPartielles[j];
			H[j*3+i] = H[i*3+j];
		}
	}
}
void hessienneComplete(uchar* topologyGrid, const uchar formId, const int nbCellsX, const int nbCellsY, const float *parametres, float *hess)
{
	for (int k=0; k<9; k++)
		hess[k] = 0;
		
	float tmp[9] = {0};
		
	for (int ix=0; ix<nbCellsX; ix++)
	{
		for (int iy=0; iy<nbCellsY; iy++)
		{
			int ind = iy*nbCellsX+ix;
			if (topologyGrid[ind] == formId)
			{
				hessienne(ix, iy, parametres, tmp);
				for (int k=0; k<9; k++)
					hess[k] += tmp[k];
			}
		}
	}
}

float gradiantFastCircle(int i, float x, float y, const float *parametres, float D)
{
	float val = deriveePartielleFastCircle(i, x, y, parametres, D);
	val *= 2*distanceSigneeFastCircle(x, y, parametres, D);
	return val;
}
void gradiantComplet(uchar* topologyGrid, const uchar formId, const int nbCellsX, const int nbCellsY, const float *parametres, float *gradiant)
{
	for (int i=0; i<3; i++)
		gradiant[i] = 0;
		
	float x, y, z, D;
	
	for (int ix=0; ix<nbCellsX; ix++)
	{
		for (int iy=0; iy<nbCellsY; iy++)
		{
			int ind = iy*nbCellsX+ix;
			if (topologyGrid[ind] == formId)
			{
				D = distanceToCenterCircle(ix, iy, parametres);
				for (int i=0; i<3; i++)
					gradiant[i] += gradiantFastCircle(i, ix, iy, parametres, D);
			}
		}
	}
}
int nbPointsActifs(uchar* topologyGrid, const uchar formId, const int nbCellsX, const int nbCellsY)
{		
	int nbPts = 0;
	for (int ix=0; ix<nbCellsX; ix++)
	{
		for (int iy=0; iy<nbCellsY; iy++)
		{
			int ind = iy*nbCellsX+ix;
			if (topologyGrid[ind] == formId)
				nbPts++;
		}
	}
	return nbPts;
}
float calculateFandGandHCircle(uchar* topologyGrid, const uchar formId, const int nbCellsX, const int nbCellsY, const float *parametres, float *G, float *H)
{
	//Calculer F au point a
	float F = calculateSumMeanSquaresCircle(topologyGrid, formId, nbCellsX, nbCellsY, parametres);
	//Calculer g et H au point a
	//   g = -0.5 * gradiant
	//   H = 0.5 * hessienne
	//Gradiant
	gradiantComplet(topologyGrid, formId, nbCellsX, nbCellsY, parametres, G);
	for (int i=0; i<3; i++) 
		G[i] *= -0.5;
	//Hessienne
	hessienneComplete(topologyGrid, formId, nbCellsX, nbCellsY, parametres, H);
	for (int k=0; k<9; k++) 
		H[k] *= 0.5;
	return F;
}
float methodeLevenbergMarquardtCircle(uchar* topologyGrid, const uchar formId, const int nbCellsX, const int nbCellsY,
	const float *parametresInitiaux, float *parametresOptimises)
{
	int nbPts = nbPointsActifs(topologyGrid, formId, nbCellsX, nbCellsY);
	if (nbPts == 0)
		return infinity;

	float dLambda = dLambdaInitial;
	float dLambdaMin = mathEpsilon;
	float dLambdaMax = infinity;
		
	float parametres[3] = {};
	for (int i=0; i<3; i++)
		parametres[i] = parametresInitiaux[i];
			
	float deltas[3] = {0};
	
	int iter = 0;
	
	float tryParam[3] = {0};
		
	bool bStop = false;

	//-------------------------------------------------------------------
	//-------------------------------------------------------------------
	//Calculer F(a initial)
	//-------------------------------------------------------------------
	//-------------------------------------------------------------------
	float F = calculateSumMeanSquaresCircle(topologyGrid, formId, nbCellsX, nbCellsY, parametres);
	float 	g[3] = {0};
	float 	H[9] = {0};
	float 	Hbarre[9] = {0};
	float	solution[3] = {0};
	
	bool bStateMoved = true;
	
	while (!bStop)
	{
		//-------------------------------------------------------------------
		//-------------------------------------------------------------------
		//Calculer g et Hbarre au point a
		//   g = -0.5 * gradiant
		//   H = 0.5 * hessienne
		//   Hbarre(j,j) = H(j,j)*(1+lambda)
		//   Hbarre(j,k) = H(j,k)
		//-------------------------------------------------------------------
		//-------------------------------------------------------------------
		if (bStateMoved)
		{
			calculateFandGandHCircle(topologyGrid, formId, nbCellsX, nbCellsY, parametres, g, H);
			//Hessienne augmentée (initialisation seulement ici)
			for (int k=0; k<9; k++)
				Hbarre[k] = H[k];
		}
		//Hessienne augmentée : ajustement des valeurs diagonales
		for (int i=0; i<3; i++)
			Hbarre[i*3+i] = H[i*3+i]*(1+dLambda);
	
		//-------------------------------------------------------------------
		//-------------------------------------------------------------------
		//Résoudre H.deltaA = g en deltaA
		//-------------------------------------------------------------------
		//-------------------------------------------------------------------
		bool bOk = solve(Hbarre, g, solution, 3);
	
		if (bOk)
		{
			for (int i=0; i<3; i++)
				deltas[i] = solution[i];
					
			//-------------------------------------------------------------------
			//-------------------------------------------------------------------
			//Calculer F(a + deltaA)
			//-------------------------------------------------------------------
			//-------------------------------------------------------------------
			for (int i=0; i<3; i++)
				tryParam[i] = parametres[i] + deltas[i];
					
			float newF = calculateSumMeanSquaresCircle(topologyGrid, formId, nbCellsX, nbCellsY, tryParam);
	
			//-------------------------------------------------------------------
			//-------------------------------------------------------------------
			//Si F(a + deltaA) >= F(a) faire
			//      lambda = 10 * lambda
			//Sinon faire
			//      lambda = 0.1 * lambda
			//      a = a + deltaA
			//-------------------------------------------------------------------
			//-------------------------------------------------------------------
			if (newF>=F)
			{
				dLambda *= lambdaScaleFactor;
				bStateMoved = false;
			}
			else
			{
				dLambda /= lambdaScaleFactor;
				bStateMoved = true;
				for (int i=0; i<3; i++)
					parametres[i] = tryParam[i];
			}
	
			//-------------------------------------------------------------------
			//-------------------------------------------------------------------
			//Critère d'arrêt
			//   Critère sur la variation de F : l’algorithme s’arrête lorsque F diminue de manière très faible
			//      delta F / Nouveau F < c 
			//      où c = 0.001 en pratique
			//   Critère sur la valeur de lambda : l’algorithme s’arrête lorsque lambda est devenu
			//      - soit très petit (<0.00000000000000000001)
			//      - soit très grand (>10000000000)
			//En pratique, on remarque que la convergence est en général atteinte rapidement (au bout de 3 ou 4 itérations)
			//-------------------------------------------------------------------
			//-------------------------------------------------------------------
			if ( (dLambda<=dLambdaMin) || (dLambda>=dLambdaMax) )		
				bStop = true;
			if (fabs(newF-F)/newF < thresholdDeltaF) 				
				bStop = true;
			if (iter>ITER_MAX) 										
				bStop = true;

			//-------------------------------------------------------------------
			//-------------------------------------------------------------------
			//Ne pas oublier de faire "F = newF" si l'état du système a changé suite à une amélioration
			//-------------------------------------------------------------------
			//-------------------------------------------------------------------
			if (bStateMoved) 
				F = newF;
		}
		else
		{
			dLambda *= lambdaScaleFactor;
			bStateMoved = false; 
		}
		iter++;
	}
	for (int i=0; i<3; i++)
		parametresOptimises[i] = parametres[i];
		
	float score = F/nbPts;
	
	return score;
}
bool passesEarlyOuts(uchar* topologyGrid, const uchar formId, const int nbCellsX, const int nbCellsY, 
	const float radiusMaxAllowed, float *boundingBox)
{
	boundingBox[0] = infinity; //xmin intentionaly initialized with positive infinity
	boundingBox[1] = infinity; //ymin intentionaly initialized with positive infinity
	boundingBox[2] = infinity_neg; //xmax intentionaly initialized with negative infinity
	boundingBox[3] = infinity_neg; //ymax intentionaly initialized with negative infinity
	
	int nbPts = 0;
	for (int ix=0; ix<nbCellsX; ix++)
	{
		for (int iy=0; iy<nbCellsY; iy++)
		{
			int ind = iy*nbCellsX+ix;
			if (topologyGrid[ind] == formId)
			{
				if (ix < boundingBox[0])
					boundingBox[0] = ix;
				if (iy < boundingBox[1])
					boundingBox[1] = iy;
				if (ix > boundingBox[2])
					boundingBox[2] = ix;
				if (iy > boundingBox[3])
					boundingBox[3] = iy;
				nbPts++;
			}
		}
	}
	if (nbPts < 3)
	return false;
	
	if (boundingBox[0] == infinity || boundingBox[1] == infinity 
		|| boundingBox[2] == infinity_neg || boundingBox[3] == infinity_neg)
		return false;
		
	float lengthX = boundingBox[2] - boundingBox[0];
	float lengthY = boundingBox[3] - boundingBox[1];
	float lengthMax = max(lengthX, lengthY);
	if (lengthMax >  2 * radiusMaxAllowed)
		return false;
	
	return true;
}

float doCircleLevenbergMarquardt(uchar* topologyGrid, const uchar formId, const int nbCellsX, const int nbCellsY, const float radiusMaxAllowed,
	float *parametresOptimises)
{
	float boundingBox[4];
	bool bOk = passesEarlyOuts(topologyGrid, formId, nbCellsX, nbCellsY, radiusMaxAllowed, boundingBox);
	if (!bOk)
		return infinity;
	
	float x0 = 0.5*(boundingBox[0]+boundingBox[2]);
	float y0 = 0.5*(boundingBox[1]+boundingBox[3]);
	float r0 = 0.5*max(boundingBox[2] - boundingBox[0], boundingBox[3] - boundingBox[1]);
	
	float parametresInitiaux[3] = { x0, y0, r0 };
	//float score = methodeLevenbergMarquardtCircle(topologyGrid, formId, nbCellsX, nbCellsY, parametresInitiaux, parametresOptimises);
	float score = infinity;
	return score;
}
