#ifndef __OPENCL_VERSION__ //This should fail during an actual openCL compile, used only to trick Eclipse into syntax highlighting this file as "C" code.
#define __kernel
#define __global
#define kernel
#define global
#define constant
#define local
#define float2 float
#define int3 int
#define float3 float
#define float4 float
#define uchar4 char
#endif

//Solves a linear system of the form A.X = b
//using Gaussian elimination with partial pivoting method.

bool solve(float* A, float* b, float* x, int n) 
{
    // Elimination
    for (int k = 0; k < n - 1; k++) 
    {
        // Trouver le pivot partiel
        int pivot_row = k;
        float max_val = fabs(A[k * n + k]);
        for (int i = k + 1; i < n; i++) 
        {
            float val = fabs(A[i * n + k]);
            if (val > max_val) 
            {
                max_val = val;
                pivot_row = i;
            }
        }

        // Vérifier si la matrice est singulière
        if (fabs(max_val) < 1e-6) 
        {
            // La matrice est singulière, gestion de l'exception
            return false;
        }

        // Échanger les lignes si nécessaire
        if (pivot_row != k) 
        {
            for (int j = k; j < n; j++) 
            {
                float temp = A[k * n + j];
                A[k * n + j] = A[pivot_row * n + j];
                A[pivot_row * n + j] = temp;
            }
            float temp = b[k];
            b[k] = b[pivot_row];
            b[pivot_row] = temp;
        }

        // Elimination
        for (int i = k + 1; i < n; i++) 
        {
            float factor = A[i * n + k] / A[k * n + k];
            for (int j = k; j < n; j++) 
                A[i * n + j] -= factor * A[k * n + j];
            b[i] -= factor * b[k];
        }
    }

    // Vérifier si la dernière valeur de la diagonale est nulle
    if (fabs(A[(n - 1) * n + (n - 1)]) < 1e-6) 
    {
        // La matrice est singulière, gestion de l'exception
        return false;
    }

    // Résolution arrière
    for (int i = n - 1; i >= 0; i--) 
    {
        float sum = 0.0;
        for (int j = i + 1; j < n; j++) 
            sum += A[i * n + j] * x[j];
        x[i] = (b[i] - sum) / A[i * n + i];
    }
    return true;
}
kernel void test_solveLinearSystem_dimension3(global const float *As, global const float *Bs, global float *clResultBuffer, const int numTasks)
{
	const int iWorker = get_global_id(0); //The worker ID
	// bound check (equivalent to the limit on a 'for' loop for standard/serial C code
	if (iWorker >= numTasks)  
		return;
		
	int startPosInAs = 9 * iWorker;
	float A[9];
	for (int k=0; k<9; k++)
		A[k] = As[startPosInAs + k];
		
	float B[3];
	int startPosInBs = 3 * iWorker;
	for (int k=0; k<3; k++)
		B[k] = Bs[startPosInBs + k];
		
	float x[3] = { infinity };

	bool bOk = solve(A, B, x, 3);
	int startPosInResult = 3 * iWorker;
	clResultBuffer[startPosInResult+0] = x[0];
	clResultBuffer[startPosInResult+1] = x[1];
	clResultBuffer[startPosInResult+2] = x[2];
}
kernel void test_solveLinearSystem_dimension4(global const float *As, global const float *Bs, global float *clResultBuffer, const int numTasks)
{
	const int iWorker = get_global_id(0); //The worker ID
	// bound check (equivalent to the limit on a 'for' loop for standard/serial C code
	if (iWorker >= numTasks)  
		return;
		
	int startPosInAs = 16 * iWorker;
	float A[16];
	for (int k=0; k<16; k++)
		A[k] = As[startPosInAs + k];
		
	float B[4];
	int startPosInBs = 4 * iWorker;
	for (int k=0; k<4; k++)
		B[k] = Bs[startPosInBs + k];
		
	float x[4] = { infinity };

	bool bOk = solve(A, B, x, 4);
	int startPosInResult = 4 * iWorker;
	clResultBuffer[startPosInResult+0] = x[0];
	clResultBuffer[startPosInResult+1] = x[1];
	clResultBuffer[startPosInResult+2] = x[2];
	clResultBuffer[startPosInResult+3] = x[3];
}
kernel void detectCircles(global const float4 *pointsBuffer, global const float4 *dirsBuffer, global float4 *clResultBuffer, 
	const float x0, const float y0, const float z0, const float sliceSize, const float sliceThickness, 
	const int numTasks, const int nPoints, const float scoreThreshold) 
{
	return;
}